// VidNexus Pro Extension - Background Service Worker

// Handle extension install
chrome.runtime.onInstalled.addListener((details) => {
    if (details.reason === 'install') {
        console.log('VidNexus Pro installed!');

        // Initialize storage
        chrome.storage.local.set({
            downloadHistory: [],
            settings: {
                quality: '720',
                format: 'h264'
            }
        });
    }
});

// Handle messages from content script or popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'DOWNLOAD_VIDEO') {
        handleDownload(message.url, message.quality || '720').then(sendResponse);
        return true; // Keep channel open for async response
    }

    if (message.type === 'FETCH_VIDEO') {
        handleFetchVideo(message.url).then(sendResponse);
        return true; // Keep channel open for async response
    }

    if (message.type === 'FETCH_PLAYLIST') {
        handleFetchPlaylist(message.url, message.playlistId).then(sendResponse);
        return true; // Keep channel open for async response
    }

    if (message.type === 'GET_CURRENT_URL') {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            sendResponse({ url: tabs[0]?.url || null });
        });
        return true;
    }
});

// Fetch video info via Cobalt (for website integration)
async function handleFetchVideo(videoUrl) {
    try {
        const response = await fetch('https://dwnld.nichind.dev/', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                url: videoUrl,
                videoQuality: '720',
                audioFormat: 'mp3',
                filenameStyle: 'basic',
                downloadMode: 'auto'
            })
        });

        if (!response.ok) {
            return { success: false, error: 'API request failed' };
        }

        const data = await response.json();
        console.log('[VidNexus Background] Cobalt response:', data);

        if (data.status === 'error') {
            return { success: false, error: data.text || 'Failed to fetch video' };
        }

        // Build video info response
        return {
            success: true,
            type: 'video',
            title: data.filename || 'Video Download',
            thumbnail: '',
            duration: 0,
            durationString: '--:--',
            uploader: 'Unknown',
            url: videoUrl,
            downloadUrl: data.url,
            qualities: [
                { quality: '1080p', label: '1080p HD' },
                { quality: '720p', label: '720p HD' },
                { quality: '480p', label: '480p' },
                { quality: '360p', label: '360p' }
            ],
            cobaltData: data
        };

    } catch (error) {
        console.error('[VidNexus Background] Fetch error:', error);
        return { success: false, error: error.message };
    }
}

// Fetch playlist videos
async function handleFetchPlaylist(playlistUrl, playlistId) {
    try {
        console.log('[VidNexus Background] Fetching playlist:', playlistId);

        // Use YouTube's RSS feed to get playlist items (public, no API key needed)
        // Format: https://www.youtube.com/feeds/videos.xml?playlist_id=PLAYLIST_ID
        const rssUrl = `https://www.youtube.com/feeds/videos.xml?playlist_id=${playlistId}`;

        const response = await fetch(rssUrl);

        if (!response.ok) {
            // Fallback: Return failure so website can show appropriate message
            return { success: false, error: 'Failed to fetch playlist data' };
        }

        const xmlText = await response.text();
        const parser = new DOMParser();
        const xmlDoc = parser.parseFromString(xmlText, 'text/xml');

        const entries = xmlDoc.querySelectorAll('entry');
        const videos = [];

        entries.forEach((entry, index) => {
            const videoId = entry.querySelector('videoId')?.textContent || '';
            const title = entry.querySelector('title')?.textContent || `Video ${index + 1}`;
            const mediaGroup = entry.querySelector('group');
            const thumbnail = `https://img.youtube.com/vi/${videoId}/mqdefault.jpg`;

            videos.push({
                id: videoId,
                title: title,
                thumbnail: thumbnail,
                duration: '--:--',
                url: `https://www.youtube.com/watch?v=${videoId}&list=${playlistId}&index=${index + 1}`
            });
        });

        if (videos.length === 0) {
            return { success: false, error: 'No videos found in playlist' };
        }

        console.log(`[VidNexus Background] Found ${videos.length} videos in playlist`);
        return { success: true, videos: videos };

    } catch (error) {
        console.error('[VidNexus Background] Playlist fetch error:', error);
        return { success: false, error: error.message };
    }
}

// Handle downloads
async function handleDownload(videoUrl, quality = '720') {
    try {
        console.log('[VidNexus] Downloading with quality:', quality);

        // Use working public Cobalt instance (no auth required)
        const response = await fetch('https://dwnld.nichind.dev/', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                url: videoUrl,
                videoQuality: quality,
                audioFormat: 'mp3',
                filenameStyle: 'basic',
                downloadMode: 'auto'
            })
        });

        if (!response.ok) {
            return { success: false, error: 'API request failed' };
        }

        const data = await response.json();

        if (data.status === 'error') {
            return { success: false, error: data.text || 'Failed to process' };
        }

        if (data.status === 'redirect' || data.status === 'stream' || data.status === 'tunnel') {
            // Start download
            chrome.downloads.download({
                url: data.url,
                filename: data.filename || 'video.mp4'
            });

            return { success: true, url: data.url };
        }

        if (data.status === 'picker' && data.picker?.length > 0) {
            chrome.downloads.download({
                url: data.picker[0].url,
                filename: data.picker[0].filename || 'video.mp4'
            });

            return { success: true, url: data.picker[0].url };
        }

        return { success: false, error: 'Unknown response format' };

    } catch (error) {
        return { success: false, error: error.message };
    }
}

// Context menu for right-click download
chrome.runtime.onInstalled.addListener(() => {
    chrome.contextMenus.create({
        id: 'vidnexus-download',
        title: 'Download with VidNexus Pro',
        contexts: ['link', 'page']
    });
});

chrome.contextMenus.onClicked.addListener((info, tab) => {
    if (info.menuItemId === 'vidnexus-download') {
        const url = info.linkUrl || info.pageUrl;
        handleDownload(url);
    }
});
