// VidNexus Pro Extension - Content Script
// Runs on VidNexus website to enable extension-powered downloads

(function () {
    // Check if we're on VidNexus website (localhost or VPS)
    const isVidNexusSite =
        window.location.hostname.includes('localhost') ||
        window.location.hostname.includes('vidnexus') ||
        window.location.hostname.includes('167.99.78.203');

    if (!isVidNexusSite) return;

    console.log('[VidNexus Extension] Detected VidNexus website, enabling extension mode');

    // Set flag that extension is installed
    localStorage.setItem('vidnexus_extension', 'true');
    localStorage.setItem('vidnexus_extension_version', '1.0.0');

    // Add a marker to the DOM for easy detection
    const marker = document.createElement('div');
    marker.id = 'vidnexus-extension-marker';
    marker.style.display = 'none';
    marker.dataset.version = '1.0.0';
    document.documentElement.appendChild(marker);

    // Send message to page that extension is ready
    window.postMessage({
        type: 'VIDNEXUS_EXTENSION_READY',
        version: '1.0.0'
    }, '*');

    // Listen for fetch and download requests from the page
    window.addEventListener('message', async (event) => {
        if (event.source !== window) return;

        // ============================================
        // Handle FETCH REQUEST - Route through background script
        // ============================================
        if (event.data.type === 'VIDNEXUS_FETCH_REQUEST') {
            console.log('[VidNexus Extension] Fetch request received:', event.data.url);

            try {
                // Send to background script (avoids CORS issues)
                chrome.runtime.sendMessage({
                    type: 'FETCH_VIDEO',
                    url: event.data.url
                }, (response) => {
                    console.log('[VidNexus Extension] Background response:', response);

                    if (response && response.success) {
                        window.postMessage({
                            type: 'VIDNEXUS_FETCH_RESPONSE',
                            ...response
                        }, '*');
                    } else {
                        window.postMessage({
                            type: 'VIDNEXUS_FETCH_RESPONSE',
                            success: false,
                            error: response?.error || 'Failed to fetch video'
                        }, '*');
                    }
                });

            } catch (error) {
                console.error('[VidNexus Extension] Fetch error:', error);
                window.postMessage({
                    type: 'VIDNEXUS_FETCH_RESPONSE',
                    success: false,
                    error: error.message
                }, '*');
            }
        }

        // ============================================
        // Handle DOWNLOAD REQUEST
        // ============================================
        if (event.data.type === 'VIDNEXUS_DOWNLOAD_REQUEST') {
            console.log('[VidNexus Extension] Download request received:', event.data.url);

            try {
                // Send download request to background script
                chrome.runtime.sendMessage({
                    type: 'DOWNLOAD_VIDEO',
                    url: event.data.url,
                    quality: event.data.quality || '720'
                }, (response) => {
                    window.postMessage({
                        type: 'VIDNEXUS_DOWNLOAD_RESPONSE',
                        requestId: event.data.requestId,
                        ...response
                    }, '*');
                });
            } catch (error) {
                window.postMessage({
                    type: 'VIDNEXUS_DOWNLOAD_RESPONSE',
                    requestId: event.data.requestId,
                    success: false,
                    error: error.message
                }, '*');
            }
        }

        // ============================================
        // Handle PLAYLIST FETCH REQUEST
        // ============================================
        if (event.data.type === 'VIDNEXUS_FETCH_PLAYLIST') {
            console.log('[VidNexus Extension] Playlist fetch request:', event.data.url);

            try {
                chrome.runtime.sendMessage({
                    type: 'FETCH_PLAYLIST',
                    url: event.data.url,
                    playlistId: event.data.playlistId
                }, (response) => {
                    window.postMessage({
                        type: 'VIDNEXUS_PLAYLIST_RESPONSE',
                        ...response
                    }, '*');
                });
            } catch (error) {
                window.postMessage({
                    type: 'VIDNEXUS_PLAYLIST_RESPONSE',
                    success: false,
                    error: error.message
                }, '*');
            }
        }
    });

    // Dispatch a custom event for easier detection
    document.dispatchEvent(new CustomEvent('vidnexus-extension-loaded', {
        detail: { version: '1.0.0' }
    }));
})();
