// VidNexus Pro Extension - Popup Script

// Working public Cobalt instance (no auth required)
const COBALT_API = 'https://dwnld.nichind.dev/';

// DOM Elements
const urlInput = document.getElementById('urlInput');
const getUrlBtn = document.getElementById('getUrlBtn');
const downloadBtn = document.getElementById('downloadBtn');
const statusArea = document.getElementById('statusArea');
const statusIcon = document.getElementById('statusIcon');
const statusTitle = document.getElementById('statusTitle');
const statusMessage = document.getElementById('statusMessage');
const downloadActions = document.getElementById('downloadActions');
const downloadLink = document.getElementById('downloadLink');
const copyBtn = document.getElementById('copyBtn');
const historyToggle = document.getElementById('historyToggle');
const historyList = document.getElementById('historyList');

// State
let currentDownloadUrl = null;
let downloadHistory = [];

// Initialize
async function init() {
    // Load history
    const stored = await chrome.storage.local.get(['downloadHistory']);
    downloadHistory = stored.downloadHistory || [];
    renderHistory();

    // Try to get URL from current tab
    getCurrentTabUrl();
}

// Get current tab URL
async function getCurrentTabUrl() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab && tab.url && isVideoUrl(tab.url)) {
            urlInput.value = tab.url;
        }
    } catch (error) {
        console.error('Failed to get tab URL:', error);
    }
}

// Check if URL might be a video URL
function isVideoUrl(url) {
    const videoSites = [
        'youtube.com', 'youtu.be',
        'tiktok.com',
        'instagram.com',
        'twitter.com', 'x.com',
        'facebook.com', 'fb.watch',
        'vimeo.com',
        'reddit.com',
        'twitch.tv',
        'dailymotion.com',
        'bilibili.com'
    ];
    return videoSites.some(site => url.includes(site));
}

// Get URL button handler
getUrlBtn.addEventListener('click', getCurrentTabUrl);

// Download button handler
downloadBtn.addEventListener('click', async () => {
    const url = urlInput.value.trim();

    if (!url) {
        showStatus('error', 'URL Required', 'Please enter a video URL');
        return;
    }

    // Validate URL
    try {
        new URL(url);
    } catch {
        showStatus('error', 'Invalid URL', 'Please enter a valid video URL');
        return;
    }

    // Start download process
    setLoading(true);
    showStatus('loading', 'Processing...', 'Fetching video information...');

    try {
        const result = await downloadWithCobalt(url);

        if (result.success) {
            currentDownloadUrl = result.url;
            downloadLink.href = result.url;
            downloadActions.classList.remove('hidden');

            showStatus('success', 'Download Ready!', result.filename || 'Click download to save the video');

            // Add to history
            addToHistory(url, result.url, result.filename);
        } else {
            showStatus('error', 'Download Failed', result.message);
            downloadActions.classList.add('hidden');
        }
    } catch (error) {
        showStatus('error', 'Error', error.message || 'Failed to process video');
        downloadActions.classList.add('hidden');
    } finally {
        setLoading(false);
    }
});

// Download with Cobalt API
async function downloadWithCobalt(videoUrl) {
    try {
        const response = await fetch(COBALT_API, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                url: videoUrl,
                videoQuality: '720',
                audioFormat: 'mp3',
                filenameStyle: 'basic',
                downloadMode: 'auto'
            })
        });

        if (!response.ok) {
            if (response.status === 429) {
                return {
                    success: false,
                    message: 'Rate limited. Please wait a moment and try again.'
                };
            }
            throw new Error(`API returned ${response.status}`);
        }

        const data = await response.json();

        if (data.status === 'error') {
            return {
                success: false,
                message: data.text || 'Failed to process video'
            };
        }

        if (data.status === 'redirect' || data.status === 'stream' || data.status === 'tunnel') {
            return {
                success: true,
                url: data.url,
                filename: data.filename || 'video'
            };
        }

        if (data.status === 'picker' && data.picker && data.picker.length > 0) {
            return {
                success: true,
                url: data.picker[0].url,
                filename: data.picker[0].filename || 'video'
            };
        }

        return {
            success: false,
            message: 'Unknown response from server'
        };

    } catch (error) {
        return {
            success: false,
            message: error.message || 'Network error'
        };
    }
}

// Copy URL button handler
copyBtn.addEventListener('click', () => {
    if (currentDownloadUrl) {
        navigator.clipboard.writeText(currentDownloadUrl).then(() => {
            copyBtn.textContent = '✓ Copied!';
            setTimeout(() => {
                copyBtn.textContent = '📋 Copy URL';
            }, 2000);
        });
    }
});

// History toggle handler
historyToggle.addEventListener('click', () => {
    historyList.classList.toggle('hidden');
});

// Show status
function showStatus(type, title, message) {
    statusArea.className = `status-area ${type}`;
    statusTitle.textContent = title;
    statusMessage.textContent = message;

    const icons = {
        loading: '⏳',
        success: '✓',
        error: '✗'
    };
    statusIcon.textContent = icons[type] || '•';
}

// Set loading state
function setLoading(loading) {
    downloadBtn.disabled = loading;
    downloadBtn.classList.toggle('loading', loading);
}

// Add to history
async function addToHistory(originalUrl, downloadUrl, filename) {
    const item = {
        originalUrl,
        downloadUrl,
        filename: filename || 'video',
        date: new Date().toISOString()
    };

    // Add to beginning
    downloadHistory.unshift(item);

    // Keep only last 10
    downloadHistory = downloadHistory.slice(0, 10);

    // Save to storage
    await chrome.storage.local.set({ downloadHistory });

    // Re-render
    renderHistory();
}

// Render history
function renderHistory() {
    if (downloadHistory.length === 0) {
        historyList.innerHTML = '<p style="text-align: center; color: rgba(255,255,255,0.3); font-size: 0.75rem; padding: 1rem;">No downloads yet</p>';
        return;
    }

    historyList.innerHTML = downloadHistory.map(item => `
    <div class="history-item">
      <span title="${item.originalUrl}">${item.filename}</span>
      <a href="${item.downloadUrl}" target="_blank">↗</a>
    </div>
  `).join('');
}

// Initialize on load
document.addEventListener('DOMContentLoaded', init);
